# SPDX-License-Identifier: LGPL-2.1-or-later

from __future__ import annotations

from Metadata import export, constmethod, sequence_protocol, class_declarations
from PyObjectBase import PyObjectBase
from typing import overload, Sequence

@export(
    TwinPointer="Vector3d",
    Include="Base/Vector3D.h",
    Constructor=True,
    Delete=True,
    NumberProtocol=True,
    RichCompare=True,
)
@sequence_protocol(
    sq_length=True,
    sq_concat=False,
    sq_repeat=False,
    sq_item=True,
    mp_subscript=True,
    sq_ass_item=True,
    mp_ass_subscript=False,
    sq_contains=False,
    sq_inplace_concat=False,
    sq_inplace_repeat=False,
)
@class_declarations(
    """public:
    VectorPy(const Vector3d & vec, PyTypeObject *T = &Type)
    :PyObjectBase(new Vector3d(vec),T){}
    VectorPy(const Vector3f & vec, PyTypeObject *T = &Type)
    :PyObjectBase(new Vector3d(vec.x,vec.y,vec.z),T){}
    Vector3d value() const
    { return *(getVectorPtr()); }
private:
    Py::List sequence;
        """
)
class Vector(PyObjectBase):
    """
    Base.Vector class.

    This class represents a 3D float vector.
    Useful to represent points in the 3D space.

    The following constructors are supported:

    Vector(x=0, y=0, z=0)
    x : float
    y : float
    z : float

    Vector(vector)
    Copy constructor.
    vector : Base.Vector

    Vector(seq)
    Define from a sequence of float.
    seq : sequence of float.

    Author: Juergen Riegel (FreeCAD@juergen-riegel.net)
    Licence: LGPL
    """

    Length: float = 0.0
    """Gets or sets the length of this vector."""

    x: float = 0.0
    """Gets or sets the X component of this vector."""

    y: float = 0.0
    """Gets or sets the Y component of this vector."""

    z: float = 0.0
    """Gets or sets the Z component of this vector."""

    # fmt: off
    @overload
    def __init__(self, x: float = 0, y: float = 0, z: float = 0) -> None: ...
    @overload
    def __init__(self, vector: "Vector") -> None: ...
    @overload
    def __init__(self, seq: Sequence[float]) -> None: ...
    # fmt: on

    @constmethod
    def __reduce__(self) -> tuple:
        """
        Serialization of Vector objects.
        """
        ...

    @constmethod
    def add(self, vector2: "Vector", /) -> "Vector":
        """
        Returns the sum of this vector and `vector2`.

        vector2 : Base.Vector
        """
        ...

    @constmethod
    def sub(self, vector2: "Vector", /) -> "Vector":
        """
        Returns the difference of this vector and `vector2`.

        vector2 : Base.Vector
        """
        ...

    @constmethod
    def negative(self) -> "Vector":
        """
        Returns the negative (opposite) of this vector.
        """
        ...

    def scale(self, x: float, y: float, z: float, /) -> "Vector":
        """
        Scales in-place this vector by the given factor in each component.

        x : float
            x-component factor scale.
        y : float
            y-component factor scale.
        z : float
            z-component factor scale.
        """
        ...

    def multiply(self, factor: float, /) -> "Vector":
        """
        Multiplies in-place each component of this vector by a single factor.
        Equivalent to scale(factor, factor, factor).

        factor : float
        """
        ...

    @constmethod
    def dot(self, vector2: "Vector", /) -> float:
        """
        Returns the scalar product (dot product) between this vector and `vector2`.

        vector2 : Base.Vector
        """
        ...

    @constmethod
    def cross(self, vector2: "Vector", /) -> "Vector":
        """
        Returns the vector product (cross product) between this vector and `vector2`.

        vector2 : Base.Vector
        """
        ...

    @constmethod
    def isOnLineSegment(self, vector1: "Vector", vector2: "Vector", /) -> bool:
        """
        Checks if this vector is on the line segment generated by `vector1` and `vector2`.

        vector1 : Base.Vector
        vector2 : Base.Vector
        """
        ...

    @constmethod
    def getAngle(self, vector2: "Vector", /) -> float:
        """
        Returns the angle in radians between this vector and `vector2`.

        vector2 : Base.Vector
        """
        ...

    def normalize(self) -> "Vector":
        """
        Normalizes in-place this vector to the length of 1.0.
        """
        ...

    @constmethod
    def isEqual(self, vector2: "Vector", tol: float = 0, /) -> bool:
        """
        Checks if the distance between the points represented by this vector
        and `vector2` is less or equal to the given tolerance.

        vector2 : Base.Vector
        tol : float
        """
        ...

    @constmethod
    def isParallel(self, vector2: "Vector", tol: float = 0, /) -> bool:
        """
        Checks if this vector and `vector2` are
        parallel less or equal to the given tolerance.

        vector2 : Base.Vector
        tol : float
        """
        ...

    @constmethod
    def isNormal(self, vector2: "Vector", tol: float = 0, /) -> bool:
        """
        Checks if this vector and `vector2` are
        normal less or equal to the given tolerance.

        vector2 : Base.Vector
        tol : float
        """
        ...

    def projectToLine(self, point: "Vector", dir: "Vector", /) -> "Vector":
        """
        Projects `point` on a line that goes through the origin with the direction `dir`.
        The result is the vector from `point` to the projected point.
        The operation is equivalent to dir_n.cross(dir_n.cross(point)), where `dir_n` is
        the vector `dir` normalized.
        The method modifies this vector instance according to result and does not
        depend on the vector itself.

        point : Base.Vector
        dir : Base.Vector
        """
        ...

    def projectToPlane(self, base: "Vector", normal: "Vector", /) -> "Vector":
        """
        Projects in-place this vector on a plane defined by a base point
        represented by `base` and a normal defined by `normal`.

        base : Base.Vector
        normal : Base.Vector
        """
        ...

    @constmethod
    def distanceToPoint(self, point2: "Vector", /) -> float:
        """
        Returns the distance to another point represented by `point2`.
        .
        point : Base.Vector
        """
        ...

    @constmethod
    def distanceToLine(self, base: "Vector", dir: "Vector", /) -> float:
        """
        Returns the distance between the point represented by this vector
        and a line defined by a base point represented by `base` and a
        direction `dir`.

        base : Base.Vector
        dir : Base.Vector
        """
        ...

    @constmethod
    def distanceToLineSegment(self, point1: "Vector", point2: "Vector", /) -> "Vector":
        """
        Returns the vector between the point represented by this vector and the point
        on the line segment with the shortest distance. The line segment is defined by
        `point1` and `point2`.

        point1 : Base.Vector
        point2 : Base.Vector
        """
        ...

    @constmethod
    def distanceToPlane(self, base: "Vector", normal: "Vector", /) -> float:
        """
        Returns the distance between this vector and a plane defined by a
        base point represented by `base` and a normal defined by `normal`.

        base : Base.Vector
        normal : Base.Vector
        """
        ...
