// SPDX-License-Identifier: LGPL-2.1-or-later

/****************************************************************************
 *   Copyright (c) 2004 Werner Mayer <wmayer[at]users.sourceforge.net>      *
 *   Copyright (c) 2023 FreeCAD Project Association                         *
 *                                                                          *
 *   This file is part of FreeCAD.                                          *
 *                                                                          *
 *   FreeCAD is free software: you can redistribute it and/or modify it     *
 *   under the terms of the GNU Lesser General Public License as            *
 *   published by the Free Software Foundation, either version 2.1 of the   *
 *   License, or (at your option) any later version.                        *
 *                                                                          *
 *   FreeCAD is distributed in the hope that it will be useful, but         *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU       *
 *   Lesser General Public License for more details.                        *
 *                                                                          *
 *   You should have received a copy of the GNU Lesser General Public       *
 *   License along with FreeCAD. If not, see                                *
 *   <https://www.gnu.org/licenses/>.                                       *
 *                                                                          *
 ***************************************************************************/


#ifndef GUI_DIALOG_PROPERTYPAGE_H
#define GUI_DIALOG_PROPERTYPAGE_H

#include <QWidget>
#include <FCGlobal.h>

namespace Gui
{
namespace Dialog
{

/** Base class for property pages.
 * \author Werner Mayer
 */
class GuiExport PropertyPage: public QWidget
{
    Q_OBJECT

public:
    explicit PropertyPage(QWidget* parent = nullptr);
    ~PropertyPage() override = default;

    bool isModified() const;
    void setModified(bool b);
    void onApply();
    void onCancel();
    void onReset();

protected:
    virtual void apply();
    virtual void cancel();
    virtual void reset();

private:
    bool bChanged; /**< for internal use only */

protected Q_SLOTS:
    virtual void loadSettings() = 0;
    virtual void saveSettings() = 0;
};

/** Base class for preferences pages.
 * \author Werner Mayer
 */
class GuiExport PreferencePage: public QWidget
{
    Q_OBJECT

public:
    explicit PreferencePage(QWidget* parent = nullptr);
    ~PreferencePage() override = default;

    bool isRestartRequired() const;
    void requireRestart();

    // this fixes issue with wordWrap on labels affecting size hints:
    // https://stackoverflow.com/questions/78276854/layout-ignoring-sizehints-when-qlabel-with-text-wrap-is-present
    bool hasHeightForWidth() const override
    {
        return false;
    }

public Q_SLOTS:
    virtual void loadSettings() = 0;
    virtual void saveSettings() = 0;
    virtual void resetSettingsToDefaults();

protected:
    void changeEvent(QEvent* event) override = 0;

private:
    bool restartRequired;
};

/** Subclass that embeds a form from a UI file.
 * \author Werner Mayer
 */
class GuiExport PreferenceUiForm: public PreferencePage
{
    Q_OBJECT

public:
    explicit PreferenceUiForm(const QString& fn, QWidget* parent = nullptr);
    ~PreferenceUiForm() override;

    void loadSettings() override;
    void saveSettings() override;

    QWidget* form();

protected:
    void changeEvent(QEvent* e) override;

private:
    template<typename PW>
    void loadPrefWidgets();
    template<typename PW>
    void savePrefWidgets();

private:
    QWidget* _form;
};

/** Base class for custom pages.
 * \author Werner Mayer
 */
class GuiExport CustomizeActionPage: public QWidget
{
    Q_OBJECT

public:
    explicit CustomizeActionPage(QWidget* parent = nullptr);
    ~CustomizeActionPage() override;

protected:
    bool event(QEvent* e) override;
    void changeEvent(QEvent* e) override = 0;

protected Q_SLOTS:
    virtual void onAddMacroAction(const QByteArray&) = 0;
    virtual void onRemoveMacroAction(const QByteArray&) = 0;
    virtual void onModifyMacroAction(const QByteArray&) = 0;
};

}  // namespace Dialog
}  // namespace Gui

#endif  // GUI_DIALOG_PROPERTYPAGE_H
