/***************************************************************************
 *   Copyright (c) 2015 Stefan Tröger <stefantroeger@gmx.net>              *
 *                                                                         *
 *   This file is part of the FreeCAD CAx development system.              *
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Library General Public           *
 *   License as published by the Free Software Foundation; either          *
 *   version 2 of the License, or (at your option) any later version.      *
 *                                                                         *
 *   This library  is distributed in the hope that it will be useful,      *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU Library General Public License for more details.                  *
 *                                                                         *
 *   You should have received a copy of the GNU Library General Public     *
 *   License along with this library; see the file COPYING.LIB. If not,    *
 *   write to the Free Software Foundation, Inc., 59 Temple Place,         *
 *   Suite 330, Boston, MA  02111-1307, USA                                *
 *                                                                         *
 ***************************************************************************/


#include <QMenu>


#include <Gui/Application.h>
#include <Gui/BitmapFactory.h>
#include <Mod/PartDesign/App/FeaturePipe.h>
#include <Mod/Part/Gui/ReferenceHighlighter.h>

#include "ViewProviderPipe.h"
#include "TaskPipeParameters.h"

using namespace PartDesignGui;

PROPERTY_SOURCE(PartDesignGui::ViewProviderPipe, PartDesignGui::ViewProvider)

ViewProviderPipe::ViewProviderPipe() = default;

ViewProviderPipe::~ViewProviderPipe() = default;

std::vector<App::DocumentObject*> ViewProviderPipe::claimChildren() const
{
    std::vector<App::DocumentObject*> temp;

    PartDesign::Pipe* pcPipe = getObject<PartDesign::Pipe>();

    App::DocumentObject* sketch = pcPipe->getVerifiedSketch(true);
    if (sketch) {
        temp.push_back(sketch);
    }

    for (App::DocumentObject* obj : pcPipe->Sections.getValues()) {
        if (obj && obj->isDerivedFrom<Part::Part2DObject>()) {
            temp.push_back(obj);
        }
    }

    App::DocumentObject* spine = pcPipe->Spine.getValue();
    if (spine && spine->isDerivedFrom<Part::Part2DObject>()) {
        temp.push_back(spine);
    }

    App::DocumentObject* auxspine = pcPipe->AuxiliarySpine.getValue();
    if (auxspine && auxspine->isDerivedFrom<Part::Part2DObject>()) {
        temp.push_back(auxspine);
    }

    return temp;
}

void ViewProviderPipe::setupContextMenu(QMenu* menu, QObject* receiver, const char* member)
{
    addDefaultAction(menu, QObject::tr("Edit Pipe"));
    PartDesignGui::ViewProvider::setupContextMenu(menu, receiver, member);
}

TaskDlgFeatureParameters* ViewProviderPipe::getEditDialog()
{
    return new TaskDlgPipeParameters(this, false);
}

void ViewProviderPipe::highlightReferences(ViewProviderPipe::Reference mode, bool on)
{
    PartDesign::Pipe* pcPipe = getObject<PartDesign::Pipe>();

    switch (mode) {
        case Spine:
            highlightReferences(
                dynamic_cast<Part::Feature*>(pcPipe->Spine.getValue()),
                pcPipe->Spine.getSubValuesStartsWith("Edge"),
                on
            );
            break;
        case AuxiliarySpine:
            highlightReferences(
                dynamic_cast<Part::Feature*>(pcPipe->AuxiliarySpine.getValue()),
                pcPipe->AuxiliarySpine.getSubValuesStartsWith("Edge"),
                on
            );
            break;
        case Profile:
            highlightReferences(
                dynamic_cast<Part::Feature*>(pcPipe->Profile.getValue()),
                pcPipe->Profile.getSubValuesStartsWith("Edge"),
                on
            );
            break;
        case Section: {
            std::vector<App::DocumentObject*> sections = pcPipe->Sections.getValues();
            for (auto it : sections) {
                highlightReferences(dynamic_cast<Part::Feature*>(it), std::vector<std::string>(), on);
            }
        } break;
        default:
            break;
    }
}

void ViewProviderPipe::highlightReferences(
    Part::Feature* base,
    const std::vector<std::string>& edges,
    bool on
)
{
    if (!base) {
        return;
    }

    PartGui::ViewProviderPart* svp = dynamic_cast<PartGui::ViewProviderPart*>(
        Gui::Application::Instance->getViewProvider(base)
    );
    if (!svp) {
        return;
    }

    std::vector<Base::Color>& edgeColors = originalLineColors[base->getID()];

    if (on) {
        if (edgeColors.empty()) {
            edgeColors = svp->LineColorArray.getValues();
            std::vector<Base::Color> colors = edgeColors;

            PartGui::ReferenceHighlighter highlighter(base->Shape.getValue(), svp->LineColor.getValue());
            highlighter.getEdgeColors(edges, colors);
            svp->LineColorArray.setValues(colors);
        }
    }
    else {
        if (!edgeColors.empty()) {
            svp->LineColorArray.setValues(edgeColors);
            edgeColors.clear();
        }
    }
}

QIcon ViewProviderPipe::getIcon() const
{
    QString str = QStringLiteral("PartDesign_");
    auto* prim = getObject<PartDesign::Pipe>();
    if (prim->getAddSubType() == PartDesign::FeatureAddSub::Additive) {
        str += QStringLiteral("Additive");
    }
    else {
        str += QStringLiteral("Subtractive");
    }

    str += QStringLiteral("Pipe.svg");
    return PartDesignGui::ViewProvider::mergeGreyableOverlayIcons(
        Gui::BitmapFactory().pixmap(str.toStdString().c_str())
    );
}
