/*
 * Copyright (c) 2016, NVIDIA CORPORATION. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "priv.h"
#include "gk20a.h"

#include <core/tegra.h>

static const struct cvb_coef gm20b_cvb_coef[] = {
	/* KHz,             c0,      c1,   c2 */
	/*  76800 */ { 1786666,  -85625, 1632 },
	/* 153600 */ { 1846729,  -87525, 1632 },
	/* 230400 */ { 1910480,  -89425, 1632 },
	/* 307200 */ { 1977920,  -91325, 1632 },
	/* 384000 */ { 2049049,  -93215, 1632 },
	/* 460800 */ { 2122872,  -95095, 1632 },
	/* 537600 */ { 2201331,  -96985, 1632 },
	/* 614400 */ { 2283479,  -98885, 1632 },
	/* 691200 */ { 2369315, -100785, 1632 },
	/* 768000 */ { 2458841, -102685, 1632 },
	/* 844800 */ { 2550821, -104555, 1632 },
	/* 921600 */ { 2647676, -106455, 1632 },
};

static const struct cvb_coef gm20b_na_cvb_coef[] = {
	/* KHz,         c0,     c1,   c2,    c3,     c4,   c5 */
	/*  76800 */ {  814294, 8144, -940, 808, -21583, 226 },
	/* 153600 */ {  856185, 8144, -940, 808, -21583, 226 },
	/* 230400 */ {  898077, 8144, -940, 808, -21583, 226 },
	/* 307200 */ {  939968, 8144, -940, 808, -21583, 226 },
	/* 384000 */ {  981860, 8144, -940, 808, -21583, 226 },
	/* 460800 */ { 1023751, 8144, -940, 808, -21583, 226 },
	/* 537600 */ { 1065642, 8144, -940, 808, -21583, 226 },
	/* 614400 */ { 1107534, 8144, -940, 808, -21583, 226 },
	/* 691200 */ { 1149425, 8144, -940, 808, -21583, 226 },
	/* 768000 */ { 1191317, 8144, -940, 808, -21583, 226 },
	/* 844800 */ { 1233208, 8144, -940, 808, -21583, 226 },
	/* 921600 */ { 1275100, 8144, -940, 808, -21583, 226 },
	/* 998400 */ { 1316991, 8144, -940, 808, -21583, 226 },
};

static const u32 speedo_to_vmin[] = {
	/*   0,      1,      2,      3,      4, */
	950000, 840000, 818750, 840000, 810000,
};

int
gm20b_volt_new(struct nvkm_device *device, enum nvkm_subdev_type type, int inst,
	       struct nvkm_volt **pvolt)
{
	struct nvkm_device_tegra *tdev = device->func->tegra(device);
	struct gk20a_volt *volt;
	u32 vmin;

	if (tdev->gpu_speedo_id >= ARRAY_SIZE(speedo_to_vmin)) {
		nvdev_error(device, "unsupported speedo %d\n",
			    tdev->gpu_speedo_id);
		return -EINVAL;
	}

	volt = kzalloc(sizeof(*volt), GFP_KERNEL);
	if (!volt)
		return -ENOMEM;
	*pvolt = &volt->base;

	vmin = speedo_to_vmin[tdev->gpu_speedo_id];

	if (tdev->gpu_speedo_id >= 1)
		return gk20a_volt_ctor(device, type, inst, gm20b_na_cvb_coef,
				       ARRAY_SIZE(gm20b_na_cvb_coef), vmin, volt);
	else
		return gk20a_volt_ctor(device, type, inst, gm20b_cvb_coef,
				       ARRAY_SIZE(gm20b_cvb_coef), vmin, volt);
}

static const struct cvb_coef gm20b_b01_na_cvb_slt_coef[] = {
	/* KHz,         c0,     c1,   c2,    c3,     c4,   c5 */
	/* 76800 */   { 590000,        0,        0,        0,        0,        0 },
	/* 153600 */  { 590000,        0,        0,        0,        0,        0 },
	/* 230400 */  { 590000,        0,        0,        0,        0,        0 },
	/* 307200 */  { 590000,        0,        0,        0,        0,        0 },
	/* 384000 */  { 590000,        0,        0,        0,        0,        0 },
	/* 460800 */  { 795089,   -11096,     -163,      298,   -10421,      162},
	/* 537600 */  { 795089,   -11096,     -163,      298,   -10421,      162 },
	/* 614400 */  { 820606,    -6285,     -452,      238,    -6182,       81 },
	/* 691200 */  { 846289,    -4565,     -552,      119,    -3958,       -2 },
	/* 768000 */  { 888720,    -5110,     -584,        0,    -2849,       39 },
	/* 844800 */  { 936634,    -6089,     -602,      -60,      -99,      -93 },
	/* 921600 */  { 982562,    -7373,     -614,     -179,     1797,      -13 },
	/* 998400 */  { 1090179,   -14125,     -497,     -179,     3518,        9 },
	/* 1075200 */ { 1155798,   -13465,     -648,        0,     1077,       40 },
	/* 1152000 */ { 1198568,   -10904,     -830,        0,     1469,      110 },
	/* 1228800 */ { 1269988,   -12707,     -859,        0,     3722,      313 },
	/* 1267200 */ { 1308155,   -13694,     -867,        0,     3681,      559 },
};

static const struct cvb_coef gm20b_b01_na_cvb_coef[] = {
	/* KHz,         c0,     c1,   c2,    c3,     c4,   c5 */
	/* 76800 */   { 610000,        0,        0,        0,        0,        0 },
	/* 153600 */  { 610000,        0,        0,        0,        0,        0 },
	/* 230400 */  { 610000,        0,        0,        0,        0,        0 },
	/* 307200 */  { 610000,        0,        0,        0,        0,        0 },
	/* 384000 */  { 610000,        0,        0,        0,        0,        0 },
	/* 460800 */  { 610000,        0,        0,        0,        0,        0 },
	/* 537600 */  { 801688,   -10900,     -163,      298,   -10599,      162 },
	/* 614400 */  { 824214,    -5743,     -452,      238,    -6325,       81 },
	/* 691200 */  { 848830,    -3903,     -552,      119,    -4030,       -2 },
	/* 768000 */  { 891575,    -4409,     -584,        0,    -2849,       39 },
	/* 844800 */  { 940071,    -5367,     -602,      -60,      -63,      -93 },
	/* 921600 */  { 986765,    -6637,     -614,     -179,     1905,      -13 },
	/* 998400 */  { 1098475,   -13529,     -497,     -179,     3626,        9 },
	/* 1075200 */ { 1163644,   -12688,     -648,        0,     1077,       40 },
	/* 1152000 */ { 1204812,    -9908,     -830,        0,     1469,      110 },
	/* 1228800 */ { 1277303,   -11675,     -859,        0,     3722,      313 },
	/* 1267200 */ { 1335531,   -12567,     -867,        0,     3681,      559 },
};

static const struct cvb_coef gm20b_b01_na_cvb_hiopt_coef[] = {
	/* KHz,         c0,     c1,   c2,    c3,     c4,   c5 */
	/* 76800 */   {   590000,        0,        0,        0,        0,        0 },
	/* 153600 */  {   590000,        0,        0,        0,        0,        0 },
	/* 230400 */  {   590000,        0,        0,        0,        0,        0 },
	/* 307200 */  {   590000,        0,        0,        0,        0,        0 },
	/* 384000 */  {   590000,        0,        0,        0,        0,        0 },
	/* 460800 */  {   590000,        0,        0,        0,        0,        0 },
	/* 537600 */  {   590000,        0,        0,        0,        0,        0 },
	/* 614400 */  {   590000,        0,        0,        0,        0,        0 },
	/* 691200 */  {   838712,    -7304,     -552,     1785,   -56250,     -450 },
	/* 768000 */  {   880210,    -7955,     -584,        0,   -42735,     8775 },
	/* 844800 */  {   926398,    -8892,     -602,     -900,    -5760,   -20925 },
	/* 921600 */  {   970060,   -10108,     -614,    -2685,    22620,    -2925 },
	/* 998400 */  {  1065665,   -16075,     -497,    -2685,    48195,     2025 },
	/* 1075200 */ {  1132576,   -16093,     -648,        0,    16155,     9000 },
	/* 1152000 */ {  1180029,   -14534,     -830,        0,    22035,    24750 },
	/* 1228800 */ {  1248293,   -16383,     -859,        0,    55830,    70425 },
};

int
gm20b_b01_volt_new(struct nvkm_device *device, enum nvkm_subdev_type type, int inst,
	       struct nvkm_volt **pvolt)
{
	struct nvkm_device_tegra *tdev = device->func->tegra(device);
	struct gk20a_volt *volt;
	u32 vmin;

	if (tdev->gpu_speedo_id >= ARRAY_SIZE(speedo_to_vmin)) {
		nvdev_error(device, "unsupported speedo %d\n",
			    tdev->gpu_speedo_id);
		return -EINVAL;
	}

	volt = kzalloc(sizeof(*volt), GFP_KERNEL);
	if (!volt)
		return -ENOMEM;
	*pvolt = &volt->base;

	vmin = speedo_to_vmin[tdev->gpu_speedo_id];

	switch (tdev->gpu_speedo_id) {
		case 3: /* HIOPT table */
			return gk20a_volt_ctor(device, type, inst, gm20b_b01_na_cvb_hiopt_coef,
						ARRAY_SIZE(gm20b_b01_na_cvb_hiopt_coef), vmin, volt);
		case 2: /* SLT table */
			return gk20a_volt_ctor(device, type, inst, gm20b_b01_na_cvb_slt_coef,
						ARRAY_SIZE(gm20b_b01_na_cvb_slt_coef), vmin, volt);
		default:
			return gk20a_volt_ctor(device, type, inst, gm20b_b01_na_cvb_coef,
						ARRAY_SIZE(gm20b_b01_na_cvb_coef), vmin, volt);
	}
}
