/*
 * Copyright 2003-2025 The IdeaVim authors
 *
 * Use of this source code is governed by an MIT-style
 * license that can be found in the LICENSE.txt file or at
 * https://opensource.org/licenses/MIT.
 */

package com.maddyhome.idea.vim.extension.nerdtree

import com.intellij.openapi.components.Service
import com.intellij.openapi.components.service
import com.intellij.ui.treeStructure.Tree
import com.maddyhome.idea.vim.extension.VimExtension
import java.awt.KeyboardFocusManager
import java.awt.event.ActionEvent
import java.awt.event.KeyEvent
import java.beans.PropertyChangeListener
import javax.swing.KeyStroke

/**
 * This plugin extends NERDTree support to components other than the Project Tool Window.
 *
 * TODO:
 * It should be considered a "sub-plugin" of NERDTree and cannot be enabled independently,
 * i.e., should not function after the NERDTree plugin is turned off.
 */
internal class NerdTreeEverywhere : VimExtension {
  companion object {
    const val PLUGIN_NAME = "NERDTreeEverywhere" // This is a temporary name
  }

  override fun getName() = PLUGIN_NAME

  val focusListener = PropertyChangeListener { evt ->
    val newFocusOwner = evt.newValue
    val oldFocusOwner = evt.oldValue
    val dispatcher = service<Dispatcher>()
    if (newFocusOwner is Tree) {
      // It's okay to have `register` called multiple times, as its internal implementation prevents duplicate registrations
      dispatcher.register(newFocusOwner)
    }
    // Unregistration of the shortcut is required to make the plugin disposable
    if (oldFocusOwner is Tree) {
      dispatcher.unregisterCustomShortcutSet(oldFocusOwner)
    }
  }

  override fun init() {
    KeyboardFocusManager.getCurrentKeyboardFocusManager().addPropertyChangeListener("focusOwner", focusListener)
  }

  @Service
  class Dispatcher : AbstractDispatcher(PLUGIN_NAME, navigationMappings.toMutableMap().apply {
    register("NERDTreeMapActivateNode", "o", NerdTreeAction { _, tree ->
      // TODO a more reliable way of invocation (such as double-clicking?)
      val listener = tree.getActionForKeyStroke(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0))
      listener.actionPerformed(ActionEvent(tree, ActionEvent.ACTION_PERFORMED, null))
    })
  }) {
    init {
      templatePresentation.isEnabledInModalContext = true
    }
  }

  override fun dispose() {
    KeyboardFocusManager.getCurrentKeyboardFocusManager().removePropertyChangeListener("focusOwner", focusListener)
    super.dispose()
  }
}
