const { parse_arg, getWindowControlLayout, getTitleFont, pangoToCss } = require("../utils/swai_utils.js");
const {ipcRenderer} = require('electron');

document.querySelector(".window-title").textContent = parse_arg("--webappName") ?? "SWAI Web App";
console.log(parse_arg("--windowId"))
let windowId = parseInt(parse_arg("--windowId")) ?? 0;

function createButton(type, iconPath) {
    const button = document.createElement("button");
    button.className = `window-button ${type} no-drag`;

    // Create an img element for the SVG
    const img = document.createElement("img");
    img.src = iconPath;
    img.draggable = false;
    img.style.width = "16px";
    img.style.height = "16px";
    img.style.color = "#ffffff";

    button.appendChild(img);
    button.onclick = () => {
        ipcRenderer.send(`${type}-${windowId}`);
    };
    return button;
}

const leftContainer = document.querySelector(".window-controls.left");
const rightContainer = document.querySelector(".window-controls.right");

function addControl(control, container) {
    switch (control) {
        case "minimize":
            container.appendChild(createButton(
                "minimize", 
                "../../icons/window-minimize-symbolic.svg"
            ));
            break;
        case "maximize":
            container.appendChild(createButton(
                "maximize",
                "../../icons/window-maximize-symbolic.svg"
            ));
            break;
        case "close":
            container.appendChild(createButton(
                "close",
                "../../icons/window-close-symbolic.svg"
            ));
            break;
    }
}

async function setupWindowControls() {
    // Parse window controls layout from URL parameters
    let leftControls = [], rightControls = [];
    try {
        const windowControls = await getWindowControlLayout();
        const fontName = await getTitleFont();
    try {
        const splitWindowControls = windowControls.split(":");
        if (splitWindowControls.length == 2) {
            leftControls = splitWindowControls[0].split(",");
            rightControls = splitWindowControls[1].split(",");
        } else {
            leftControls = [];
            rightControls = "minimize,maximize,close".split(",");
        }
    } catch (e) {
        leftControls = [];
        rightControls = "minimize,maximize,close".split(",");
    }
    let { fontFamily, fontWeight, fontStyle, fontSize } = pangoToCss(fontName);
    document.querySelector(".window-title").style.fontFamily = fontFamily;
    document.querySelector(".window-title").style.fontWeight = "Bold";
    document.querySelector(".window-title").style.fontStyle = fontStyle;
    document.querySelector(".window-title").style.fontSize = fontSize;

    leftControls.forEach(control => addControl(control, leftContainer));
    rightControls.forEach(control => addControl(control, rightContainer));

    let back_button = createButton("back", "../../icons/go-previous-symbolic.svg");
    back_button.classList.add("flat");
    back_button.disabled = true;

    let forward_button = createButton("forward", "../../icons/go-next-symbolic.svg");
    forward_button.classList.add("flat");
    forward_button.disabled = true;

    if (leftContainer.children.length <= rightContainer.children.length) {
        leftContainer.append(back_button);
        leftContainer.append(forward_button);
    } else {
        rightContainer.prepend(back_button);
        rightContainer.prepend(forward_button);
    }

    } catch (error) {
        console.error("Error setting up window controls:", error);
        // Fallback to default layout
        let leftControls = [];
        let rightControls = ["minimize", "maximize", "close"];
        
        leftControls.forEach(control => addControl(control, leftContainer));
        rightControls.forEach(control => addControl(control, rightContainer));
        
        let back_button = createButton("back", "../../icons/go-previous-symbolic.svg");
        back_button.classList.add("flat");
        back_button.disabled = true;

        let forward_button = createButton("forward", "../../icons/go-next-symbolic.svg");
        forward_button.classList.add("flat");
        forward_button.disabled = true;

        if (leftContainer.children.length <= rightContainer.children.length) {
            leftContainer.append(back_button);
            leftContainer.append(forward_button);
        } else {
            rightContainer.prepend(back_button);
            rightContainer.prepend(forward_button);
        }
    }
}

setupWindowControls().catch(error => {
    console.error("Failed to setup window controls:", error);
});

/**
 * Set the titlebar background color dynamically
 * @param {string|null} color - Hex color string (e.g., "#ffffff") or null for default
 * @param {boolean|null} isLight - Whether the color is light (for text contrast) or null for auto-detect
 */
function set_color_titlebar(color, isLight) {
    const root = document.documentElement;
    
    if (!color) {
        // Reset to default colors
        root.style.removeProperty('--headerbar-bg');
        root.style.removeProperty('--headerbar-fg');
        document.body.classList.remove('custom-theme-color', 'theme-light', 'theme-dark');
        return;
    }
    
    // Set the custom background color
    root.style.setProperty('--headerbar-bg', color);
    
    // Use pure black or white for maximum contrast
    const textColor = isLight ? '#000000' : '#ffffff';
    root.style.setProperty('--headerbar-fg', textColor);
    
    // Add classes for CSS to know we're using a custom theme
    document.body.classList.add('custom-theme-color');
    document.body.classList.remove('theme-light', 'theme-dark');
    document.body.classList.add(isLight ? 'theme-light' : 'theme-dark');
}

// Handle window title updates
ipcRenderer.on(`url-changed-${windowId}`, (event, title, can_back, can_forward) => {
    document.querySelector(".window-title").textContent = title;
    document.querySelector(".window-button.back").disabled = !can_back;
    document.querySelector(".window-button.forward").disabled = !can_forward;
});

ipcRenderer.on(`window-state-change-${windowId}`, (event, isMaximized) => {
// Handle window state changes for maximize/restore button
    const maximizeButtons = document.querySelectorAll(".window-button.maximize");
    maximizeButtons.forEach(button => {
        button.innerHTML = isMaximized ? restoreSVG : maximizeSVG;
        button.title = isMaximized ? "Restore" : "Maximize";
    });
});

// Handle theme color updates from the main web content
ipcRenderer.on(`theme-color-changed-${windowId}`, (event, themeData) => {
    if (themeData && themeData.color) {
        set_color_titlebar(themeData.color, themeData.isLight);
    } else {
        // Reset to default
        set_color_titlebar(null, null);
    }
});