const {
    BaseWindow,
    BrowserWindow,
    WebContentsView,
    session,
    ipcMain,
} = require("electron");
const fs = require("fs");
const path = require("path");
const os = require("os");
const { parse_arg, openUrlInBrowser, wildCardMatchList } = require("../utils/swai_utils");
const Ecosystem = require("../utils/Ecosystem");

class WindowManager {
    constructor(debug_mode = false, ecosystemId = null, swaiRegistry = null) {
        this.windowId = 0;
        this.debug_mode = debug_mode;
        this.ecosystemId = ecosystemId;
        this.swaiRegistry = swaiRegistry;
        this.ecosystemManager = null; // Will be set by EcosystemManager
        this.dataPathsInitialized = false; // Track if app data paths have been set
        this.loginModeByWindow = new Map(); // Track login mode state per window (windowId -> boolean)
    }
    
    setEcosystemManager(ecosystemManager) {
        this.ecosystemManager = ecosystemManager;
    }

    createWindow(webapp) {
        this.windowId += 1;
        let boundsTimeout = null;
        let webapp_ecosystem = null;

        // Use ecosystem ID for data path so apps in same ecosystem share data/cookies
        const dataId = this.ecosystemId || webapp.app_id;
        
        // Only set app data paths once per process (subsequent calls can cause issues)
        if (!this.dataPathsInitialized) {
            let data_path = path.join(os.homedir(), '.local', 'share', 'swai', 'app_data', dataId);
            const { app } = require("electron");
            app.setPath("userData", path.join(data_path, "userData"));
            app.setPath("appData", path.join(data_path, "appData"));
            app.setPath("sessionData", path.join(data_path, "sessionData"));
            app.setPath("cache", path.join(data_path, "cache"));
            app.setPath("logs", path.join(data_path, "logs"));
            app.setPath("crashDumps", path.join(data_path, "crashDumps"));
            this.dataPathsInitialized = true;
            console.log(`Initialized app data paths for: ${dataId}`);
        }

        // Use ecosystem ID for session partition so apps in same ecosystem share sessions/cookies
        let webAppSession = session.fromPartition(`persist:${dataId}`);

        let thisWindow = null;

        // For ecosystem apps, use internal title for DBus registration
        let windowTitle = webapp.app_name;
        let internalTitle = null;
        let currentDisplayTitle = webapp.app_name; // Track current display title to avoid redundant DBus calls
        
        if (webapp.ecosystem && this.swaiRegistry) {
            // Always generate internal title for ecosystem apps, even if DBus not available yet
            internalTitle = this.swaiRegistry.generateInternalTitle(webapp.app_id, this.windowId);
            windowTitle = internalTitle; // Set window title to internal title for DBus registration
        }

        thisWindow = new BaseWindow({
            width: 1200,
            height: 900,
            title: windowTitle,
            frame: false,
            show: true, // Explicitly show the window
        });
        
        console.log(`Created window ${this.windowId} for ${webapp.app_name}`);

        const set_title = (title) => {
            if (internalTitle && this.swaiRegistry && this.swaiRegistry.isAvailable()) {
                // For ecosystem apps, keep window title as internal title but update DBus display title
                thisWindow.title = internalTitle;
                
                // Only update DBus if the display title has actually changed
                if (title !== currentDisplayTitle) {
                    currentDisplayTitle = title;
                    this.swaiRegistry.updateTitle(internalTitle, title).catch(error => {
                        console.warn('Failed to update DBus title registration:', error.message);
                    });
                }
            } else {
                // For non-ecosystem apps, set title normally
                thisWindow.title = title;
            }
        };

        let titlebar = new WebContentsView({
            webPreferences: {
                contextIsolation: false,
                nodeIntegration: true,
                additionalArguments: [
                    `--windowId=${this.windowId}`,
                    `--webappName=${webapp.app_name}`,
                ],
            }
        });

        let mainWebContents = new WebContentsView({
            nodeIntegration: false,
            contextIsolation: true,
            webPreferences: {
                session: webAppSession
            }
        });

        titlebar.webContents.loadURL(
            `file://${path.join(__dirname, "../components/titlebar.html")}`,
        );
        thisWindow.removeMenu();
        thisWindow.contentView.addChildView(titlebar);
        thisWindow.contentView.addChildView(mainWebContents);

        let custom_main_url = parse_arg("--custom_main_url");
        if (custom_main_url) {
            webapp.main_url = custom_main_url;
            if (!wildCardMatchList(custom_main_url, webapp.allowed_urls)) {
                webapp.allowed_urls.push(custom_main_url);
            }
        }

        mainWebContents.webContents.loadURL(webapp.main_url);

        if (this.debug_mode) {
            titlebar.webContents.openDevTools();
            mainWebContents.webContents.openDevTools();
        }

        const updateBounds = () => {
            const { width, height } = thisWindow.getBounds();
            let titlebarHeight = this.debug_mode ? 300 : 36;
            titlebar.setBounds({ x: 0, y: 0, width, height: titlebarHeight });
            mainWebContents.setBounds({
                x: 0,
                y: titlebarHeight,
                width,
                height: height - titlebarHeight,
            });
        };

        updateBounds();

        thisWindow.on("resize", () => {
            updateBounds();
            if (boundsTimeout) {
                clearTimeout(boundsTimeout);
            }
            boundsTimeout = setTimeout(updateBounds, 1000);
        });

        this._setupMainWebContents(mainWebContents, thisWindow, webapp, set_title, webAppSession, titlebar);
        this._setupIpcHandlers(thisWindow, mainWebContents);
        this._setupContextMenu(mainWebContents);

        // Register the window with DBus for custom WM_CLASS handling (ecosystem apps only)
        if (webapp.ecosystem && this.swaiRegistry) {
            // Ensure SwaiRegistry is initialized before attempting registration
            // Don't await this - let it happen asynchronously so window creation isn't blocked
            this._registerWindowWithDelay(webapp, internalTitle, currentDisplayTitle, thisWindow).catch(err => {
                console.error('Window registration failed:', err);
            });
        }

        // Clean up login mode state when window closes
        const windowIdForCleanup = this.windowId;
        thisWindow.on('closed', () => {
            this.loginModeByWindow.delete(windowIdForCleanup);
            console.log(`Cleaned up login mode state for window ${windowIdForCleanup}`);
        });

        // Ensure window is visible
        thisWindow.show();
        console.log(`✓ Window ${this.windowId} for ${webapp.app_name} created and shown`);

        return thisWindow;
    }

    async _registerWindowWithDelay(webapp, internalTitle, currentDisplayTitle, thisWindow) {
        try {
            console.log(`Starting registration for window ${this.windowId} - ${webapp.app_name}`);
            
            // Ensure SwaiRegistry is initialized
            if (!this.swaiRegistry.initializationComplete) {
                console.log(`Waiting for SwaiRegistry initialization for ${webapp.app_name}...`);
                await this.swaiRegistry.initialize();
            }

            // Generate internal title if not already done
            if (!internalTitle) {
                internalTitle = this.swaiRegistry.generateInternalTitle(webapp.app_id, this.windowId);
                thisWindow.title = internalTitle;
                console.log(`Generated internal title: ${internalTitle}`);
            }

            if (internalTitle && this.swaiRegistry.isAvailable()) {
                const success = await this.swaiRegistry.registerTitle(internalTitle, currentDisplayTitle);
                if (success) {
                    console.log(`✓ Registered ${webapp.app_name} (window ${this.windowId}) with custom WM_CLASS: ${webapp.app_id}`);
                } else {
                    console.warn(`Failed to register ${webapp.app_name} with SwaiRegistry`);
                }
            } else {
                console.log(`DBus not available, skipping registration for ${webapp.app_name}`);
            }
        } catch (error) {
            console.error('Error during window registration:', error);
            // Don't let registration errors prevent the window from working
        }
    }

    _setupMainWebContents(mainWebContents, thisWindow, webapp, set_title, webAppSession, titlebar) {
        // Set up theme color listener once (outside dom-ready to avoid duplicate listeners)
        if (webapp.use_dynamic_titlebar_colors !== false) {
            this._setupThemeColorListener(mainWebContents, titlebar);
        }
        
        mainWebContents.webContents.on("dom-ready", () => {
            console.log("dom ready");

            if (thisWindow) {
                const cssPath = path.join(__dirname, "../../system_theme.css");
                fs.readFile(cssPath, "utf-8", (err, data) => {
                    if (err) {
                        console.error("Failed to read CSS file:", err);
                        return;
                    }
                    if (thisWindow) {
                        mainWebContents.webContents.insertCSS(data).catch((error) => {
                            console.error("Failed to inject CSS:", error);
                        });
                    }
                });
                
                // Inject theme color detector script (needs to be done on each navigation)
                if (webapp.use_dynamic_titlebar_colors !== false) {
                    this._injectThemeColorDetectorScript(mainWebContents);
                }
            }

            // Store windowId in closure for use in handlers
            const currentWindowId = this.windowId;
            
            mainWebContents.webContents.setWindowOpenHandler((details) => {
                console.log("Attempted new window:", details.url);
                console.log("Current URL:", mainWebContents.webContents.getURL());
                console.log("Multi-window allowed:", webapp.allow_multi_window);
                
                const navResult = this._get_navigation_allowed(
                    mainWebContents.webContents.getURL(),
                    details.url,
                    webapp,
                    currentWindowId
                );
                
                if (!navResult.allowed) {
                    console.log(`[SWAI] New window blocked - reason: ${navResult.reason}`);
                    console.log(`[SWAI]   URL: ${details.url}`);
                    console.log(`[SWAI]   Ecosystem app available: ${navResult.ecosystemApp?.app_name || 'none'}`);
                    
                    // Check if an ecosystem app can handle this URL
                    if (navResult.ecosystemApp && this.ecosystemManager) {
                        console.log(`[SWAI] Opening in ecosystem app: ${navResult.ecosystemApp.app_name}`);
                        this.ecosystemManager.handleNewAppRequest({
                            ...navResult.ecosystemApp,
                            main_url: details.url
                        });
                        return { action: "deny" };
                    }
                    
                    console.log("[SWAI] No ecosystem app found, opening in system browser via xdg-open");
                    openUrlInBrowser(details.url);
                    return { action: "deny" };
                }
                
                // Navigation is allowed - check how to handle the new window
                
                // Check if URL matches allowed_urls for new window handling
                const urlMatchesAllowed = wildCardMatchList(details.url, webapp.allowed_urls);
                
                if (urlMatchesAllowed && webapp.allow_multi_window) {
                    // allow_multi_window is true and URL matches allowed - create new app window
                    console.log("URL matches allowed list and multi-window enabled, creating new window");
                    try {
                        let new_window_app = webapp.clone();
                        new_window_app.main_url = details.url;
                        const newWindow = this.createWindow(new_window_app);
                        
                        // Track the new window in the ecosystem manager
                        if (this.ecosystemManager) {
                            this.ecosystemManager.trackWindow(new_window_app.app_id, newWindow);
                            console.log("New window tracked in ecosystem manager");
                        }
                        
                        console.log("New window created successfully");
                        return { action: "deny" };
                    } catch (error) {
                        console.error("Error creating new window:", error);
                        return { action: "deny" };
                    }
                } else {
                    // Either allow_multi_window is false OR URL doesn't match allowed_urls
                    // Use transient modal window
                    console.log("Creating transient window (multi-window disabled or URL not in allowed list)");
                    this.createTransientWindow(details.url, webAppSession, thisWindow);
                    return { action: "deny" };
                }
            });

            this._setupNavigationHandlers(mainWebContents, webapp, set_title, titlebar, currentWindowId);
            this._setupKeyboardShortcuts(mainWebContents);
        });
    }

    _setupNavigationHandlers(mainWebContents, webapp, set_title, titlebar, windowId) {
        mainWebContents.webContents.on("will-navigate", (event, navigationUrl) => {
            const navResult = this._get_navigation_allowed(
                mainWebContents.webContents.getURL(),
                navigationUrl,
                webapp,
                windowId
            );
            
            if (!navResult.allowed) {
                event.preventDefault();
                console.log(`[SWAI] Navigation blocked - reason: ${navResult.reason}`);
                console.log(`[SWAI]   URL: ${navigationUrl}`);
                console.log(`[SWAI]   Ecosystem app available: ${navResult.ecosystemApp?.app_name || 'none'}`);
                
                // Check if an ecosystem app can handle this URL
                if (navResult.ecosystemApp && this.ecosystemManager) {
                    console.log(`[SWAI] Opening in ecosystem app: ${navResult.ecosystemApp.app_name}`);
                    this.ecosystemManager.handleNewAppRequest({
                        ...navResult.ecosystemApp,
                        main_url: navigationUrl
                    });
                } else {
                    console.log("[SWAI] No ecosystem app found, opening in system browser via xdg-open");
                    openUrlInBrowser(navigationUrl);
                }
            }
        });

        // Handle redirects - catch URLs that redirect to not_allowed_urls
        mainWebContents.webContents.on("will-redirect", (event, redirectUrl, isMainFrame) => {
            if (!isMainFrame) return; // Only check main frame redirects
            
            const currentUrl = mainWebContents.webContents.getURL();
            const navResult = this._get_navigation_allowed(
                currentUrl,
                redirectUrl,
                webapp,
                windowId
            );
            
            if (!navResult.allowed) {
                event.preventDefault();
                console.log(`[SWAI] Redirect blocked - reason: ${navResult.reason}`);
                console.log(`[SWAI]   Redirect URL: ${redirectUrl}`);
                console.log(`[SWAI]   From: ${currentUrl}`);
                
                // Check if an ecosystem app can handle this URL
                if (navResult.ecosystemApp && this.ecosystemManager) {
                    console.log(`[SWAI] Opening redirect in ecosystem app: ${navResult.ecosystemApp.app_name}`);
                    this.ecosystemManager.handleNewAppRequest({
                        ...navResult.ecosystemApp,
                        main_url: redirectUrl
                    });
                } else {
                    console.log("[SWAI] Opening redirect in system browser");
                    openUrlInBrowser(redirectUrl);
                }
            }
        });

        mainWebContents.webContents.on("did-navigate", (event, navigationUrl) => {
            set_title(mainWebContents.webContents.getTitle());
            this._sendUrlChanged(mainWebContents, titlebar);
            
            // Update login mode based on the URL we landed on
            this._updateLoginModeAfterNavigation(windowId, navigationUrl, webapp);
        });

        mainWebContents.webContents.on("did-navigate-in-page", (event, navigationUrl) => {
            set_title(mainWebContents.webContents.getTitle());
            this._sendUrlChanged(mainWebContents, titlebar);
            
            // Update login mode based on the URL we landed on
            this._updateLoginModeAfterNavigation(windowId, navigationUrl, webapp);
        });

        mainWebContents.webContents.on("page-title-updated", (event, title) => {
            set_title(mainWebContents.webContents.getTitle());
            this._sendUrlChanged(mainWebContents, titlebar);
        });
    }

    _sendUrlChanged(mainWebContents, titlebar) {
        if (titlebar) {
            titlebar.webContents.send(
                `url-changed-${this.windowId}`,
                mainWebContents.webContents.getTitle(),
                mainWebContents.webContents.navigationHistory.canGoBack(),
                mainWebContents.webContents.navigationHistory.canGoForward(),
            );
        }
    }

    /**
     * Set up the one-time listener for theme color updates from the detector
     * This should be called once per window, not on every dom-ready
     */
    _setupThemeColorListener(mainWebContents, titlebar) {
        const windowId = this.windowId;
        
        // Listen for theme color updates via console messages
        mainWebContents.webContents.on('console-message', (event, level, message) => {
            if (message.startsWith('__SWAI_THEME_COLOR__')) {
                try {
                    const themeData = JSON.parse(message.slice('__SWAI_THEME_COLOR__'.length));
                    if (titlebar && !titlebar.webContents.isDestroyed()) {
                        titlebar.webContents.send(`theme-color-changed-${windowId}`, themeData);
                    }
                } catch (parseError) {
                    console.error("Failed to parse theme color data:", parseError);
                }
            }
        });
    }

    /**
     * Inject the theme color detector script into the web contents
     * This needs to be called on each dom-ready as the page context changes
     */
    _injectThemeColorDetectorScript(mainWebContents) {
        const detectorPath = path.join(__dirname, "../components/theme-color-detector.js");
        
        fs.readFile(detectorPath, "utf-8", (err, detectorCode) => {
            if (err) {
                console.error("Failed to read theme color detector:", err);
                return;
            }
            
            // Set up callback that uses console.log to communicate back
            // (works with contextIsolation: true)
            const callbackSetup = `
                window.__swai_theme_color_callback = (themeData) => {
                    console.log('__SWAI_THEME_COLOR__' + JSON.stringify(themeData));
                };
            `;
            
            mainWebContents.webContents.executeJavaScript(callbackSetup + detectorCode).catch(error => {
                console.error("Failed to inject theme color detector:", error);
            });
        });
    }

    _setupKeyboardShortcuts(mainWebContents) {
        mainWebContents.webContents.on("before-input-event", (event, input) => {
            if (input.control) {
                if (input.key === "I" && input.control && input.shift) {
                    mainWebContents.webContents.toggleDevTools();
                    event.preventDefault();
                } else if (input.key === "+") {
                    mainWebContents.webContents.setZoomFactor(
                        mainWebContents.webContents.getZoomFactor() + 0.1,
                    );
                    event.preventDefault();
                } else if (input.key === "-") {
                    mainWebContents.webContents.setZoomFactor(
                        mainWebContents.webContents.getZoomFactor() - 0.1,
                    );
                    event.preventDefault();
                } else if (input.key === "0") {
                    mainWebContents.webContents.setZoomFactor(1.0);
                    event.preventDefault();
                } else if (input.key == "r") {
                    mainWebContents.webContents.reload();
                    event.preventDefault();
                }
            }
        });
    }

    _setupIpcHandlers(thisWindow, mainWebContents) {
        ipcMain.on(`back-${this.windowId}`, () => {
            if (thisWindow) {
                mainWebContents.webContents.navigationHistory.goBack();
            }
        });

        ipcMain.on(`forward-${this.windowId}`, () => {
            if (thisWindow) {
                mainWebContents.webContents.navigationHistory.goForward();
            }
        });

        ipcMain.on(`close-${this.windowId}`, () => {
            if (thisWindow) {
                thisWindow.close();
            }
        });

        ipcMain.on(`minimize-${this.windowId}`, () => {
            if (thisWindow) {
                thisWindow.minimize();
            }
        });

        ipcMain.on(`maximize-${this.windowId}`, () => {
            if (thisWindow) {
                if (thisWindow.isMaximized()) {
                    thisWindow.unmaximize();
                } else {
                    thisWindow.maximize();
                }
            }
        });
    }

    async _setupContextMenu(mainWebContents) {
        try {
            const contextMenu = await import("electron-context-menu");
            const dispose = contextMenu.default({
                window: mainWebContents.webContents,
                showInspectElement: false,
            });
        } catch (error) {
            console.error("Failed to setup context menu:", error);
        }
    }

    /**
     * Check if login mode is active for a window
     * Login mode allows navigation to any URL (for external auth like Okta)
     */
    _isLoginModeActive(windowId) {
        return this.loginModeByWindow.get(windowId) === true;
    }

    /**
     * Enable login mode for a window
     */
    _enableLoginMode(windowId) {
        if (!this.loginModeByWindow.get(windowId)) {
            console.log(`Login mode ENABLED for window ${windowId}`);
            this.loginModeByWindow.set(windowId, true);
        }
    }

    /**
     * Disable login mode for a window
     */
    _disableLoginMode(windowId) {
        if (this.loginModeByWindow.get(windowId)) {
            console.log(`Login mode DISABLED for window ${windowId}`);
            this.loginModeByWindow.set(windowId, false);
        }
    }

    /**
     * Update login mode based on current URL after navigation completes
     * Login mode is disabled when we land on an allowed URL that is NOT a login URL
     */
    _updateLoginModeAfterNavigation(windowId, currentUrl, webapp) {
        const isOnLoginUrl = wildCardMatchList(currentUrl, webapp.login_urls);
        const isOnAllowedUrl = wildCardMatchList(currentUrl, webapp.allowed_urls);
        
        if (isOnLoginUrl) {
            // We're on a login URL - enable login mode
            this._enableLoginMode(windowId);
        } else if (isOnAllowedUrl) {
            // We're on an allowed URL that is NOT a login URL - disable login mode
            this._disableLoginMode(windowId);
        }
        // If neither login nor allowed URL, maintain current state (could be mid-OAuth flow)
    }

    /**
     * Check if a URL path suggests it will redirect to a not_allowed domain
     * For example: https://www.notion.so/calendarAuth -> https://calendar.notion.so/*
     * Returns true if the URL path contains keywords that match blocked subdomains
     */
    _isRedirectToBlockedDomain(url, notAllowedUrls) {
        try {
            const urlObj = new URL(url);
            const pathname = urlObj.pathname.toLowerCase();
            
            // Extract subdomain keywords from not_allowed_urls
            // e.g., "calendar.notion.so" -> "calendar", "mail.notion.so" -> "mail"
            for (const pattern of notAllowedUrls) {
                try {
                    // Extract domain from pattern (handle wildcards)
                    const patternUrl = pattern.replace(/\*/g, '');
                    const patternObj = new URL(patternUrl);
                    const subdomain = patternObj.hostname.split('.')[0];
                    
                    // Check if pathname contains this subdomain keyword
                    if (subdomain && pathname.includes(subdomain)) {
                        console.log(`[SWAI] URL path "${pathname}" suggests redirect to blocked domain "${subdomain}"`);
                        return true;
                    }
                } catch (e) {
                    // Skip invalid patterns
                }
            }
        } catch (e) {
            // URL parsing failed, skip check
        }
        return false;
    }

    /**
     * Check if a URL can be handled by another app in the same ecosystem
     * Returns the webapp that can handle it, or null
     */
    _findEcosystemAppForUrl(url, currentWebapp) {
        if (!this.ecosystemManager || !currentWebapp.ecosystem) {
            return null;
        }
        
        const ecosystemApps = this.ecosystemManager.getApps();
        for (const app of ecosystemApps) {
            // Skip the current app
            if (app.app_id === currentWebapp.app_id) {
                continue;
            }
            // Check if this app can handle the URL
            if (wildCardMatchList(url, app.allowed_urls) && 
                !wildCardMatchList(url, app.not_allowed_urls)) {
                return app;
            }
        }
        return null;
    }

    /**
     * Determines if navigation should be allowed within the app
     * 
     * Rules:
     * 1. not_allowed_urls: Always blocked - open in browser or ecosystem app
     * 2. Login mode active: Allow all navigation (for external auth flows like Okta)
     * 3. Current URL is a login URL: Enable login mode and allow navigation
     * 4. Navigation URL matches allowed_urls: Allow
     * 5. Otherwise: Block and open in browser or ecosystem app
     * 
     * @param {string} currentUrl - The current page URL
     * @param {string} navigationUrl - The URL being navigated to
     * @param {object} webapp - The webapp configuration
     * @param {number} windowId - The window ID for login mode tracking
     * @returns {object} { allowed: boolean, reason: string, ecosystemApp: WebApp|null }
     */
    _get_navigation_allowed(currentUrl, navigationUrl, webapp, windowId = null) {
        console.log(`[SWAI] Checking navigation: ${currentUrl} -> ${navigationUrl}`);
        console.log(`[SWAI]   Window ID: ${windowId}, Login mode: ${windowId !== null ? this._isLoginModeActive(windowId) : 'N/A'}`);
        console.log(`[SWAI]   not_allowed_urls: ${JSON.stringify(webapp.not_allowed_urls)}`);
        
        // Rule 1: Check not_allowed_urls first (highest priority block)
        const matchesNotAllowed = wildCardMatchList(navigationUrl, webapp.not_allowed_urls);
        console.log(`[SWAI]   Matches not_allowed_urls: ${matchesNotAllowed}`);
        
        // Also check if URL path suggests it will redirect to a blocked domain
        // e.g., https://www.notion.so/calendarAuth -> https://calendar.notion.so/*
        const isRedirectToBlocked = this._isRedirectToBlockedDomain(navigationUrl, webapp.not_allowed_urls);
        console.log(`[SWAI]   Suggests redirect to blocked domain: ${isRedirectToBlocked}`);
        
        if (matchesNotAllowed || isRedirectToBlocked) {
            console.log(`[SWAI] BLOCKED by not_allowed_urls${isRedirectToBlocked ? ' (redirect-prone URL)' : ''}`);
            // Check if an ecosystem app can handle this URL
            const ecosystemApp = this._findEcosystemAppForUrl(navigationUrl, webapp);
            return { 
                allowed: false, 
                reason: 'not_allowed_urls', 
                ecosystemApp 
            };
        }

        // Rule 2: If login mode is active, allow all navigation
        if (windowId !== null && this._isLoginModeActive(windowId)) {
            console.log(`[SWAI] ALLOWED by login mode`);
            return { allowed: true, reason: 'login_mode_active', ecosystemApp: null };
        }

        // Rule 3: Check if currently on a login URL - enable login mode and allow
        if (wildCardMatchList(currentUrl, webapp.login_urls)) {
            if (windowId !== null) {
                this._enableLoginMode(windowId);
            }
            console.log(`[SWAI] ALLOWED - on login URL, login mode enabled`);
            return { allowed: true, reason: 'on_login_url', ecosystemApp: null };
        }

        // Rule 4: Check if navigation URL matches allowed_urls
        if (wildCardMatchList(navigationUrl, webapp.allowed_urls)) {
            console.log(`[SWAI] ALLOWED by allowed_urls`);
            return { allowed: true, reason: 'allowed_urls', ecosystemApp: null };
        }

        // Rule 5: Navigation not allowed
        console.log(`[SWAI] BLOCKED - not in allowed_urls`);
        // Check if an ecosystem app can handle this URL
        const ecosystemApp = this._findEcosystemAppForUrl(navigationUrl, webapp);
        return { 
            allowed: false, 
            reason: 'not_in_allowed_urls', 
            ecosystemApp 
        };
    }

    createTransientWindow(url, session, parent) {
        let transientWindow = new BrowserWindow({
            width: 800,
            height: 600,
            parent: parent,
            modal: true,
            title: "SWAI Web App",
            webPreferences: {
                session: session,
            },
        });
        transientWindow.loadURL(url);
        transientWindow.once("ready-to-show", () => {
            transientWindow.show();
        });
        transientWindow.removeMenu();
    }
}

module.exports = WindowManager; 