#include "BallScrollBar.h"

#include <QPainter>
#include <QPaintEvent>
#include <QStyleOptionSlider>

void BallScrollBar::setBallVisible(bool visible)
{
  ball_visible = visible;
  update();
}

void BallScrollBar::setBallColor(QColor const & color)
{
  ball_color = color;
  update();
}

void BallScrollBar::setBallValue(int value)
{
  ball_value = value;
  update();
}

void BallScrollBar::mouseReleaseEvent(QMouseEvent * event)
{
  QScrollBar::mouseReleaseEvent(event);
  if (event->button() == Qt::LeftButton) { emit scrollingFinished(); }
}

void BallScrollBar::keyReleaseEvent(QKeyEvent * event)
{
  QScrollBar::keyReleaseEvent(event);

  // Ignore key release events generated by auto-repeat
  if (event->isAutoRepeat()) { return; }

  switch (event->key())
  {
  case Qt::Key_Left:
  case Qt::Key_Right:
  case Qt::Key_Up:
  case Qt::Key_Down:
  case Qt::Key_PageUp:
  case Qt::Key_PageDown:
  case Qt::Key_Home:
  case Qt::Key_End:
    emit scrollingFinished();
    break;
  }
}

#ifndef QT_NO_WHEELEVENT
void BallScrollBar::wheelEvent(QWheelEvent * event)
{
  QScrollBar::wheelEvent(event);
  emit scrollingFinished();
}
#endif

#ifndef QT_NO_CONTEXTMENU
void BallScrollBar::contextMenuEvent(QContextMenuEvent * event)
{
  QScrollBar::contextMenuEvent(event);
  emit scrollingFinished();
}
#endif

void BallScrollBar::paintEvent(QPaintEvent * event)
{
  QScrollBar::paintEvent(event);

  // Only draw the ball if the scroll bar is enabled, the ball is allowed to be
  // visible, and the ball position is within bounds.
  if (isEnabled() && ball_visible &&
    (ball_value >= minimum()) && (ball_value <= maximum()))
  {
    QStyleOptionSlider opt;
    initStyleOption(&opt);

    QRect groove_rect = style()->subControlRect(QStyle::CC_ScrollBar, &opt,
      QStyle::SC_ScrollBarGroove, this);
    QRect slider_rect = style()->subControlRect(QStyle::CC_ScrollBar, &opt,
      QStyle::SC_ScrollBarSlider, this);

    int span, length_offset, thickness_offset, radius;
    QPainter painter(this);

    if (orientation() == Qt::Horizontal)
    {
      span = groove_rect.width() - slider_rect.width();
      length_offset = groove_rect.x() + slider_rect.width() / 2;
      thickness_offset = height() / 2;
      radius = height() / 4;
    }
    else
    {
      span = groove_rect.height() - slider_rect.height();
      length_offset = groove_rect.y() + slider_rect.height() / 2;
      thickness_offset = width() / 2;
      radius = width() / 4;

      painter.rotate(90);
      painter.scale(1, -1);
    }
    int pos = QStyle::sliderPositionFromValue(minimum(), maximum(),
      ball_value, span);

    painter.setRenderHint(QPainter::Antialiasing);
    painter.setPen(Qt::NoPen);
    painter.setBrush(ball_color);
    painter.translate(length_offset, thickness_offset);
    painter.drawEllipse(QPoint(pos, 0), radius, radius);
  }
}
